##########################################################################################
########################################## INFO ##########################################
##########################################################################################

"""
This program is provided during session 2 of the PyRat project:
https://hub.imt-atlantique.fr/ueinfo-fise1a/s5/project/session2/practical/
"""

##########################################################################################
######################################### IMPORTS ########################################
##########################################################################################

# External imports
import abc

# PyRat imports
from pyrat import Graph

##########################################################################################
######################################### CLASSES ########################################
##########################################################################################

class Traversal (abc.ABC):

    """
    This is an abstract class that defines the interface for graph traversal algorithms.
    It is meant to be inherited by specific traversal algorithm implementations, such as breadth-first search (BFS) or depth-first search (DFS).
    
    The class defines three abstract methods that must be implemented in the subclasses:
        * ``initialize_structure``: Initializes the data structure needed for the traversal.
        * ``add_to_structure``: Adds an element to the data structure.
        * ``get_from_structure``: Extracts an element from the data structure.
    The choice of data structure (e.g., queue, stack) will determine the behavior of the traversal algorithm.
    """

    ##################################################################################
    #                                   CONSTRUCTOR                                  #
    ##################################################################################

    def __init__ ( self,
                   *args:    object,
                   **kwargs: object
                 ) ->        None:

        """
        *(This class is abstract and meant to be subclassed, not instantiated directly).*

        Initializes a new instance of the class.

        Args:
            args:   Arguments to pass to the parent constructor.
            kwargs: Keyword arguments to pass to the parent constructor.
        """
        
        # Inherit from parent class
        super().__init__(*args, **kwargs)

        # Calling traversal will set these attributes
        self.routing_table = None
        self.distances = None

    ##################################################################################
    #                                     METHODS                                    #
    ##################################################################################

    def traversal ( self,
                    graph:  Graph,
                    source: int
                  ) ->      None:

        """
        This method performs a traversal of the given graph from the given source vertex.
        It computes the distances from the source to all other vertices, as well as a routing table that can be used to reconstruct paths.
        The results are stored in the ``distances`` and ``routing_table`` attributes.
        
        Args:
            graph:  The graph to traverse.
            source: The source vertex of the traversal (we type it as int here because we will only manipulate mazes, but Graph uses Hashable as a more generic type).
        """

        # Initialization
        self.distances = {}
        self.routing_table = {}

        # Complete the method here
        # Use methods `initialize_structure(...)`, `add_to_structure(...)` and `get_from_structure(...)` to manage the data structure

    ##################################################################################

    def find_route ( self,
                     target: int
                   ) ->      list[int]:

        """
        This method finds the route from the source vertex to a target vertex using a routing table.

        Args:
            target: The target vertex.

        Returns:
            The route from the source to the target as a list of vertices.
        """

        # Debug
        assert self.routing_table is not None, "Traversal must be performed before finding a route."
        assert target in self.routing_table, "Target vertex not encountered during traversal."
        
        # Your code here
        # Use the `routing_table` attribute to find the route from the source (with parent None) to the target

    ##################################################################################

    @abc.abstractmethod
    def initialize_structure (self) -> object:
    
        """
        *(This method is abstract and must be implemented in the subclasses).*

        It initializes the data structure needed for the traversal.

        Returns:
            The initialized data structure.

        Raises:
            NotImplementedError: If the method is not implemented in the subclass.
        """

        # This method must be implemented in the child classes
        # By default we raise an error
        raise NotImplementedError("This method must be implemented in the child classes.")

    ##################################################################################

    @abc.abstractmethod
    def add_to_structure ( self,
                           structure: object,
                           element:   object
                         ) ->         object:
    
        """
        *(This method is abstract and must be implemented in the subclasses).*

        It adds an element to the data structure and returns the updated data structure.

        Returns:
            The updated data structure.

        Raises:
            NotImplementedError: If the method is not implemented in the subclass.
        """

        # This method must be implemented in the child classes
        # By default we raise an error
        raise NotImplementedError("This method must be implemented in the child classes.")

    ##################################################################################

    @abc.abstractmethod
    def get_from_structure ( self,
                             structure: object,
                           ) ->         tuple[object, object]:

        """
        *(This method is abstract and must be implemented in the subclasses).*

        It extracts an element from the data structure and returns it along with the updated data structure.

        Returns:
            A tuple containing the extracted element and the updated data structure.

        Raises:
            NotImplementedError: If the method is not implemented in the subclass.
        """

        # This method must be implemented in the child classes
        # By default we raise an error
        raise NotImplementedError("This method must be implemented in the child classes.")

##########################################################################################
##########################################################################################
